(* ::Subsubsection:: *)
(*WrightCamera package*)

(* ::Text:: *)
(*This impements the Wright Camera optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/WrightCamera"}]*)

BeginPackage["GeometricOptics`Packages`WrightCamera`", {"GeometricOptics`"}]

Options[WrightCamera] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
	CameraType can be
					"SD" Houghton's telescope with symmetric doublet
					"DDR" Houghton's telescope with doublet with different radii - no control of astigmatism
*)

WrightCamera::usage="WrightCamera calculates parameters for a Houghton-Cassegrain telescope optical system.";
WrightCameraInputPanel::usage = "WrightCameraInputPanel gives the GUI panel to work with WrightCamera";

$WrightCameraInputVariables = {"f1", "Ni", "thick", "diam", "\[CapitalDelta]", "\[Theta]"};

Begin["`Private`"]

WrightCamera[{f1_, Ni_, thick_, diam_, delta_, theta_}, opts___]:= WrightCamera[f1, Ni, thick, diam, delta, theta];

WrightCamera[f1_, Ni_, thick_, diam_, delta_, theta_, OptionsPattern[]] := 
Quiet@Module[{\[Alpha], \[Delta], a4, K, fun, sag, inputs, outputs, inPanel, outPanel, title},
 		If[ArgumentsQ["WrightCamera", {$WrightCameraInputVariables, {f1, Ni, thick, diam, delta, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 4, "PackageName" -> "WrightCamera", "Arguments" -> <|"f1" -> f1, "Ni" -> Ni, "thick" -> thick, "diam" -> diam, "\[CapitalDelta]" -> delta, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
			
			\[Alpha] = 0.866;
			\[Delta] = delta/f1;
			a4 = 1/(16 f1^3 (-1 + Ni) \[Delta]);
			K = -1 + 2/\[Delta];
			TotalAberrations[{10^10, -(1/(a4 \[Alpha]^2 diam^2)), -2 f1}, {thick, \[Delta] f1}, {{1, Ni, 1, -1}}, {0, {a4, as}, K}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];
			fun = -(1/2) \[Alpha]^2*a4*diam^2*Global`y^2 + a4*Global`y^4;
			sag = fun /. {Global`y -> \[Alpha] diam/2};

			outputs = {	
						{"Conic constant of mirror", N[K]},
						{"Corrector equation", fun},
						{"Curvature radius of the second corrector surface", -(1/(a4 (\[Alpha]^2) (diam^2) ))},
						{"Corrector sagitta:", sag},
						{"Corrector plot", Plot[fun, {Global`y, -(diam/2), diam/2}, PlotRange -> All, AspectRatio -> 0.2]},
						{"Image height", GOimageHeight[[1, GOn]]},
						{"Spherical aberration", GOaberration[[1]]},
						{"Sagittal coma", GOcoma[[1]]},
						{"Astigmatism", GOastigmatism[[1]]},
						{"Petzval radius", -f1},
						{"Sagittal curvature radius", (4 f1)/(\[Delta] (-4 + \[Delta] + K \[Delta]))},
						{"Meridional curvature radius", (4 f1)/(8 - 12 \[Delta] + 3 (1 + K) \[Delta]^2)},
						{"Mean curvature", 1/2 ( (4 f1)/(\[Delta] (-4 + \[Delta] + K \[Delta])) + (4 f1)/(8 - 12 \[Delta] + 3 (1 + K) \[Delta]^2))},
						{"Focal length", GOfocalLength[[1]]}};
			
			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Refractive index of corrector", "Ni", Ni},
							{"List of thickness of the lenses", "thick", thick},
							{"Diameter of the corrector", "diam", diam},
							{"Distance between the corrector and the mirror", "\[CapitalDelta]", delta},
							{"Field angle  in degree", "\[Theta]", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];

			title = $GeometricOpticsPackagesList[SelectFirst[#PackageName == "WrightCamera" &], "Description"];
			
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> title,																	
												"date" -> DateString[], 
												"function" -> "WrightCamera", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> title, 
													"date" -> DateString[], 
													"function" -> "WrightCamera", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1", f1},
														 {"Ni", Ni},
														 {"thick", thick},
														 {"diam", diam},
														 {"\[CapitalDelta]", delta},
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["WrightCamera not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

WrightCameraInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "WrightCamera"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, Ni, thick, diam, delta, theta} = ReplaceAll[$WrightCameraInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run WrightCamera function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the focal length of the primary mirror"]},
						  {"Refractive index of the corrector", "Ni", Tooltip[InputField[Dynamic[Ni], Alignment -> Center], "Input the refractive index of the corrector"]},
						  {"List of thickness of the lenses", "thick", Tooltip[InputField[Dynamic[thick], Alignment -> Center], "Insert the list of thickness of the lenses"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Distance between the corrector and the mirror", "\[CapitalDelta]", Tooltip[InputField[Dynamic[delta], Alignment -> Center], "Input the distance between the corrector and the mirror"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, Ni, thick, diam, delta, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, Ni, thick, diam, delta, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]